package ru.masterdata.storage.user;

import org.keycloak.models.*;
import ru.masterdata.utils.MD5Utils;
import org.keycloak.credential.CredentialInput;

import java.util.Collections;
import java.util.Map;
import java.util.ArrayList;
import java.util.List;

import org.keycloak.models.cache.CachedUserModel;
import ru.masterdata.model.MdmUserModel;
import ru.masterdata.model.UserAdapter;
import org.keycloak.storage.StorageId;

import ru.masterdata.utils.DataSourceProvider;
import org.keycloak.component.ComponentModel;
import org.jboss.logging.Logger;
import org.keycloak.models.cache.OnUserCache;
import org.keycloak.credential.CredentialInputValidator;
import org.keycloak.storage.user.UserQueryProvider;
import org.keycloak.storage.user.UserLookupProvider;
import org.keycloak.storage.UserStorageProvider;

public class MdmUserStorageProvider implements UserStorageProvider, UserLookupProvider, UserQueryProvider, CredentialInputValidator, OnUserCache {

    private static final Logger logger = Logger.getLogger(MdmUserStorageProvider.class);
    public static final String PASSWORD_CACHE_KEY = UserAdapter.class.getName() + ".password";
    private final ComponentModel model;
    private final KeycloakSession session;
    private final DataSourceProvider dataSourceProvider;

    public MdmUserStorageProvider(final ComponentModel model, final KeycloakSession session, final DataSourceProvider dataSourceProvider) {
        this.model = model;
        this.session = session;
        this.dataSourceProvider = dataSourceProvider;
    }

    @Override
    public void preRemove(final RealmModel realm) {
    }

    @Override
    public void preRemove(final RealmModel realm, final GroupModel group) {
    }

    @Override
    public void preRemove(final RealmModel realm, final RoleModel role) {
    }

    @Override
    public UserModel getUserById(final String id, final RealmModel realm) {
        MdmUserStorageProvider.logger.info("getUserById: " + id);
        final String sid = StorageId.externalId(id);
        final MdmUserModel user = this.dataSourceProvider.getUserByLogin(sid);
        if (user == null) {
            MdmUserStorageProvider.logger.info("could not find user by id: " + id);
            return null;
        }
        return new UserAdapter(this.session, realm, this.model, user, this.dataSourceProvider);
    }

    public UserModel getUserByUsername(final String username, final RealmModel realm) {
        MdmUserStorageProvider.logger.info("getUserByUsername: " + username);
        final MdmUserModel user = this.dataSourceProvider.getUserByLogin(username);
        if (user == null) {
            MdmUserStorageProvider.logger.info("could not find username: " + username);
            return null;
        }
        return new UserAdapter(this.session, realm, this.model, user, this.dataSourceProvider);
    }

    @Override
    public UserModel getUserByEmail(final String email, final RealmModel realm) {
        final MdmUserModel user = this.dataSourceProvider.getUserByEmail(email);
        if (user == null) {
            return null;
        }
        return new UserAdapter(this.session, realm, this.model, user, this.dataSourceProvider);
    }

    //    @Override
//    public void onCache(final RealmModel realm, final CachedUserModel user, final UserModel delegate) {
//        final String password = ((UserAdapter) delegate).getPassword();
//        if (password != null) {
//            user.getCachedWith().put(MdmUserStorageProvider.PASSWORD_CACHE_KEY, password);
//        }
//    }
    @Override
    public void onCache(RealmModel realm, CachedUserModel user, UserModel delegate) {
        String password = ((UserAdapter) delegate).getPassword();
        if (password != null) {
            user.getCachedWith().put(PASSWORD_CACHE_KEY, password);
        }
    }

    @Override
    public int getUsersCount(final RealmModel realm) {
        return this.dataSourceProvider.getUserCount();
    }

    @Override
    public List<UserModel> getUsers(final RealmModel realm) {
        return this.toUserModel(dataSourceProvider.getAllUsers(), realm);
    }

    private List<UserModel> toUserModel(final List<MdmUserModel> users, final RealmModel realm) {
        final List<UserModel> res = new ArrayList<UserModel>();
        for (final MdmUserModel user : users) {
            res.add(new UserAdapter(session, realm, model, user, dataSourceProvider));
        }
        return res;
    }

    public List<UserModel> getUsers(final RealmModel realm, final int firstResult, final int maxResults) {
        final List<MdmUserModel> users = dataSourceProvider.getAllUsers(firstResult, maxResults);
        return this.toUserModel(users, realm);
    }

    public List<UserModel> searchForUser(final String search, final RealmModel realm) {
        return this.searchForUser(search, realm, -1, -1);
    }

    public List<UserModel> searchForUser(final String search, final RealmModel realm, final int firstResult, final int maxResults) {
        final List<MdmUserModel> results = dataSourceProvider.searchUserByLoginOrEmail(search, firstResult, maxResults);
        return this.toUserModel(results, realm);
    }

    public List<UserModel> searchForUser(final Map<String, String> params, final RealmModel realm) {
        return Collections.emptyList();
    }

    public List<UserModel> searchForUser(final Map<String, String> params, final RealmModel realm, final int firstResult, final int maxResults) {
        return Collections.emptyList();
    }

    public List<UserModel> getGroupMembers(final RealmModel realm, final GroupModel group, final int firstResult, final int maxResults) {
        return Collections.emptyList();
    }

    public List<UserModel> getGroupMembers(final RealmModel realm, final GroupModel group) {
        return Collections.emptyList();
    }

    @Override
    public List<UserModel> getRoleMembers(final RealmModel realm, final RoleModel role, final int firstResult, final int maxResults) {
        return Collections.emptyList();
    }

    @Override
    public List<UserModel> getRoleMembers(final RealmModel realm, final RoleModel role) {
        return Collections.emptyList();
    }

    @Override
    public List<UserModel> searchForUserByUserAttribute(final String attrName, final String attrValue, final RealmModel realm) {
        return Collections.emptyList();
    }

    @Override
    public boolean supportsCredentialType(final String credentialType) {
        return "password".equals(credentialType);
    }

    @Override
    public boolean isConfiguredFor(final RealmModel realmModel, final UserModel userModel, final String credentialType) {
        return this.supportsCredentialType(credentialType);
    }

    public String getPassword(UserModel user) {
        String password = null;
        if (user instanceof CachedUserModel) {
            password = (String) ((CachedUserModel) user).getCachedWith().get(PASSWORD_CACHE_KEY);
        } else if (user instanceof UserAdapter) {
            password = ((UserAdapter) user).getPassword();
        }
        return password;
    }

    @Override
    public boolean isValid(RealmModel realm, UserModel user, CredentialInput input) {
        if (!supportsCredentialType(input.getType()) || !(input instanceof UserCredentialModel)) return false;
        UserCredentialModel cred = (UserCredentialModel) input;
        String password = getPassword(user);
        return password != null &&
                MD5Utils.getHexString(cred.getValue()).equalsIgnoreCase(password);
        // password.equals(cred.getValue());
    }


    public boolean isValidOld(final RealmModel realmModel, final UserModel userModel, final CredentialInput credentialInput) {
        if (!this.supportsCredentialType(credentialInput.getType())) {
            MdmUserStorageProvider.logger.debugv("Credential type \"{0}\" is not supported", credentialInput.getType());
            return false;
        }
        final String passwdHash = ((UserAdapter) userModel).getPassword();
        final String password = credentialInput.getChallengeResponse();
        if (password == null) {
            MdmUserStorageProvider.logger.debugv("Credential type null is not supported");
            return false;
        }
        return MD5Utils.getHexString(password).equalsIgnoreCase(passwdHash);
    }


    @Override
    public void close() {

    }
}