package ru.masterdata.utils;

import ru.masterdata.model.MdmRoleModel;
import java.util.HashSet;
import org.keycloak.models.RoleModel;
import java.util.Set;
import org.keycloak.models.RoleContainerModel;
import java.util.ArrayList;
import java.util.List;
import java.sql.PreparedStatement;
import java.sql.Connection;
import java.sql.SQLException;
import ru.masterdata.model.MdmUserModel;

import java.sql.ResultSet;
import javax.sql.DataSource;

public class DataSourceProvider {
    private static final int COL_LOGIN = 1;
    private static final int COL_MAIL = 2;
    private static final int COL_PASSWD = 3;
    private static final int COL_PHONE = 4;
    private final DataSource dataSource;
    private static final String USER_ALL = "select login,email,passwd,phone from MDM_CL_USER ";
    private static final String USER_ALL_WINDOW = "select login,email,passwd,phone from MDM_CL_USER  offset ? limit ?";
    private static final String USER_COUNT = "select count(*) from MDM_CL_USER ";
    private static final String USER_BY_LOGIN = "select login,email,passwd,phone from MDM_CL_USER  where lower(login)=?";
    private static final String USER_BY_EMAIL = "select login,email,passwd,phone from MDM_CL_USER  where lower(email)=?";
    private static final String USER_SEARCH = "select login,email,passwd,phone from MDM_CL_USER  where ( lower(login) like ? or email like ? ) order by login offset ? limit ?";
    private static final String USER_ROLE = "select role_code from mdm_user_role_view where lower(login)=?";

    public DataSourceProvider(final DataSource dataSource) {
        this.dataSource = dataSource;
    }

    private MdmUserModel toModel(final ResultSet resultSet) throws SQLException {
        return new MdmUserModel(resultSet.getString(COL_LOGIN), resultSet.getString(COL_MAIL), resultSet.getString(COL_PASSWD), resultSet.getString(COL_PHONE));
    }

    private MdmUserModel getUserFromQry(final String sql, final String login) {
        try (final Connection connection = this.dataSource.getConnection();

             final PreparedStatement stmt = connection.prepareStatement(sql)) {
            stmt.setString(COL_LOGIN, login.toLowerCase());
            final ResultSet resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                return toModel(resultSet);
            }
            return null;

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public MdmUserModel getUserByLogin(final String login) {
        return this.getUserFromQry(USER_BY_LOGIN, login.toLowerCase());
    }

    public MdmUserModel getUserByEmail(final String mail) {
        return this.getUserFromQry(USER_BY_EMAIL, mail);
    }


    public int getUserCount() {
        try (final Connection connection = this.dataSource.getConnection(); final PreparedStatement stmt = connection.prepareStatement(USER_COUNT)) {
            final ResultSet resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                return resultSet.getInt(1);
            }

            return 0;
        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public List<MdmUserModel> getAllUsers() {
        try (final Connection connection = dataSource.getConnection(); final PreparedStatement stmt = connection.prepareStatement(USER_ALL)) {
            final ResultSet resultSet = stmt.executeQuery();
            final List<MdmUserModel> result = new ArrayList<MdmUserModel>();
            while (resultSet.next()) {
                result.add(this.toModel(resultSet));
            }
            return result;
        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public List<MdmUserModel> getAllUsers(final int first, final int max) {
        try (final Connection connection = this.dataSource.getConnection(); final PreparedStatement stmt = connection.prepareStatement(USER_ALL_WINDOW)) {
            stmt.setInt(1, first);
            stmt.setInt(2, max);
            final ResultSet resultSet = stmt.executeQuery();
            final List<MdmUserModel> result = new ArrayList<MdmUserModel>();
            while (resultSet.next()) {
                result.add(this.toModel(resultSet));
            }
            return result;

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public Set<RoleModel> getUserRoles(final String login, final RoleContainerModel realm) {

        try (final Connection connection = this.dataSource.getConnection(); PreparedStatement stmt = connection.prepareStatement(USER_ROLE)) {
            stmt.setString(COL_LOGIN, login.toLowerCase());
            final ResultSet resultSet = stmt.executeQuery();
            final Set<RoleModel> result = new HashSet<RoleModel>();
            while (resultSet.next()) {
                result.add(new MdmRoleModel(realm, resultSet.getString(COL_LOGIN)));
            }
            return result;

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public List<MdmUserModel> searchUserByLoginOrEmail(final String search, final int firstResult, final int maxResults) {
        try (Connection connection = dataSource.getConnection(); PreparedStatement stmt = connection.prepareStatement(USER_SEARCH)) {
            final String searchParam = "%" + search.toLowerCase() + "%";
            stmt.setString(COL_LOGIN, searchParam);
            stmt.setString(COL_MAIL, searchParam);
            stmt.setInt(3, (firstResult != -1) ? firstResult : 0);
            stmt.setInt(4, (maxResults != -1) ? maxResults : Integer.MAX_VALUE);
            final ResultSet resultSet = stmt.executeQuery();
            final List<MdmUserModel> result = new ArrayList<>();
            while (resultSet.next()) {
                result.add(this.toModel(resultSet));
            }
            return result;
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

    }
}